//

// Cryptoki interface file

//

// Please refer to the RSA Laboratories PKCS #11: Cryptographic Token Interface

// Standard for details

//

// Copyright 1997 Datakey, Inc.

// -----------------------------------------------------------------------------



#ifndef CRYPTOKI_H

#define CRYPTOKI_H



#if CRYPTOKI_VERSION > 1



#include "pkcs11.h"



#else



#ifdef WIN32

#pragma pack(push,cryptoki,1)

#endif



#ifndef FALSE

#define FALSE	0

#endif



#ifndef TRUE

#define TRUE	(!FALSE)

#endif



// Base types ------------------------------------------------------------------



typedef unsigned char CK_BYTE;			// an unsigned 8-bit value

typedef CK_BYTE CK_CHAR;				// an unsigned 8-bit character

typedef CK_BYTE CK_BBOOL;				// a BYTE-sized Boolean flag

typedef unsigned short int CK_USHORT;	// an unsigned value, at least 16 bits

typedef unsigned long int CK_ULONG;		// an unsigned value, at least 32 bits

typedef CK_ULONG CK_FLAGS;				// at least 32 bits, each bit is boolean



typedef CK_BYTE * CK_BYTE_PTR;			// pointer to CK_BYTE

typedef CK_CHAR * CK_CHAR_PTR;			// pointer to CK_CHAR

typedef CK_USHORT * CK_USHORT_PTR;		// pointer to CK_USHORT

typedef void * CK_VOID_PTR;				// pointer to a void

#ifndef NULL_PTR

#define NULL_PTR	0					// a null pointer

#endif



// General information types ---------------------------------------------------



// Cryptoki version

//

typedef struct CK_VERSION

{

	CK_BYTE major;

	CK_BYTE minor;

} CK_VERSION;



// Cryptoki general information

//

typedef struct CK_INFO

{

	CK_VERSION version;

	CK_CHAR manufacturerID[32];

	CK_FLAGS flags;

} CK_INFO;

typedef CK_INFO * CK_INFO_PTR;



// Notification 

//

typedef unsigned long CK_NOTIFICATION;



#define CKN_SURRENDER		0x0u

#define CKN_COMPLETE		0x1u

#define CKN_DEVICE_REMOVED	0x2u

#define CKN_TOKEN_INSERTION 0x3u



//  Slot and token types -------------------------------------------------------



typedef CK_ULONG CK_SLOT_ID;		// slot id

typedef CK_SLOT_ID * CK_SLOT_ID_PTR;



// Slot information

//

typedef struct CK_SLOT_INFO

{

	CK_CHAR slotDescription[64];

	CK_CHAR manufacturerID[32];

	CK_FLAGS flags;

} CK_SLOT_INFO;

typedef struct CK_SLOT_INFO * CK_SLOT_INFO_PTR;



// Slot information flags

//

enum

{

	CKF_TOKEN_PRESENT = 1,		// TRUE if a token is present in the slot

	CKF_REMOVABLE_DEVICE= 2,	// TRUE if the reader supports removable devices

	CKF_HW_SLOT = 4,			// TRUE if the slot is a hardware slot

};



// Token information

//

typedef struct CK_TOKEN_INFO

{

	CK_CHAR label[32];

	CK_CHAR manufacturerID[32];

	CK_CHAR model[16];

	CK_CHAR serialNumber[16];

	CK_FLAGS flags;

	CK_USHORT usMaxSessionCount;

	CK_USHORT usSessionCount;

	CK_USHORT usMaxRwSessionCount;

	CK_USHORT usRwSessionCount;

	CK_USHORT usMaxPinLen;

	CK_USHORT usMinPinLen;

	CK_ULONG ulTotalPublicMemory;

	CK_ULONG ulFreePublicMemory;

	CK_ULONG ulTotalPrivateMemory;

	CK_ULONG ulFreePrivateMemory;

} CK_TOKEN_INFO;

typedef CK_TOKEN_INFO * CK_TOKEN_INFO_PTR;



// Token information flags

//

enum

{

	CKF_RNG = 1,					// TRUE if token has random number generator

	CKF_WRITE_PROTECTED = 2,		// TRUE if the token is write-protected

	CKF_LOGIN_REQUIRED = 4,			// TRUE if logon required for crypto ops

	CKF_USER_PIN_INITIALIZED = 8,	// TRUE if normal user's PIN is initialized

	CKF_EXCLUSIVE_EXISTS = 0x10,	// TRUE if an exclusive session exists

};



// Session Types ---------------------------------------------------------------



typedef CK_ULONG CK_SESSION_HANDLE;	// session identifier

typedef CK_SESSION_HANDLE * CK_SESSION_HANDLE_PTR;



#ifdef ENTRUST

// Cryptoki user types

//

#define CKU_SO   0		// Security Officer

#define CKU_USER 1		// Normal user

#define CKU_DSO  2		// Datakey Security Officer



typedef CK_USHORT CK_USER_TYPE;



// Session state

//

#define CKS_RO_PUBLIC_SESSION 0

#define CKS_RO_USER_FUNCTIONS 1

#define CKS_RW_PUBLIC_SESSION 2

#define CKS_RW_USER_FUNCTIONS 3

#define CKS_RW_SO_FUNCTIONS   4



typedef CK_USHORT CK_STATE;



#else

// Cryptoki user types

//

typedef enum CK_USER_TYPE

{

	CKU_SO,			// Security Officer

	CKU_USER,		// Normal user

	CKU_DSO,		// Datakey Security Officer

} CK_USER_TYPE;



// Session state

//

typedef enum CK_STATE

{

	CKS_RO_PUBLIC_SESSION,

	CKS_RO_USER_FUNCTIONS,

	CKS_RW_PUBLIC_SESSION,

	CKS_RW_USER_FUNCTIONS,

	CKS_RW_SO_FUNCTIONS

} CK_STATE;



#endif

// Session information

//

typedef struct CK_SESSION_INFO

{

	CK_SLOT_ID slotID;

	CK_STATE state;

	CK_FLAGS flags;

	CK_USHORT usDeviceError;

} CK_SESSION_INFO;

typedef CK_SESSION_INFO *CK_SESSION_INFO_PTR;



// Session information flags

//

enum

{

	CKF_EXCLUSIVE_SESSION = 1,			// TRUE if the session is exclusive

	CKF_RW_SESSION = 2,					// TRUE if the session is read/write

	CKF_SERIAL_SESSION = 4				// TRUE if funcions are synchronous

};



// Object types ----------------------------------------------------------------



typedef CK_ULONG CK_OBJECT_HANDLE;		// object identifier

typedef CK_OBJECT_HANDLE * CK_OBJECT_HANDLE_PTR;

typedef CK_USHORT CK_OBJECT_CLASS;		// class identifier

typedef CK_OBJECT_CLASS * CK_OBJECT_CLASS_PTR;



// Object classes

//

#define CKO_DATA 			0x0u

#define CKO_CERTIFICATE 	0x1u

#define CKO_PUBLIC_KEY 		0x2u

#define CKO_PRIVATE_KEY 	0x3u

#define CKO_SECRET_KEY 		0x4u

#define CKO_VENDOR_DEFINED 	0x8000u



typedef CK_USHORT CK_KEY_TYPE;			// key type identifier



// Key types

//

#define CKK_RSA				0x0u

#define CKK_DSA				0x1u

#define CKK_DH				0x2u

#define CKK_GENERIC_SECRET	0x10u

#define CKK_RC2				0x11u

#define CKK_RC4				0x12u

#define CKK_DES				0x13u

#define CKK_DES2			0x14u

#define CKK_DES3			0x15u

#define CKK_VENDOR_DEFINED	0x8000u



typedef CK_USHORT CK_CERTIFICATE_TYPE;



// Certificate types

//

#define CKC_X_509			0u

#define CKC_VENDOR_DEFINED	0x8000u



// Attribute types

//

typedef CK_USHORT CK_ATTRIBUTE_TYPE;

#define CKA_CLASS				0x0u

#define CKA_TOKEN				0x1u

#define CKA_PRIVATE				0x2u

#define CKA_LABEL				0x3u

#define CKA_APPLICATION			0x10u

#define CKA_VALUE				0x11u

#define CKA_CERTIFICATE_TYPE	0x80u

#define CKA_ISSUER				0x81u

#define CKA_SERIAL_NUMBER		0x82u

#define CKA_KEY_TYPE			0x100u

#define CKA_SUBJECT				0x101u

#define CKA_ID					0x102u

#define CKA_SENSITIVE			0x103u

#define CKA_ENCRYPT				0x104u

#define CKA_DECRYPT				0x105u

#define CKA_WRAP				0x106u

#define CKA_UNWRAP				0x107u

#define CKA_SIGN				0x108u

#define CKA_SIGN_RECOVER		0x109u

#define CKA_VERIFY				0x10au

#define CKA_VERIFY_RECOVER		0x10bu

#define CKA_DERIVE				0x10cu

#define CKA_START_DATE			0x110u

#define CKA_END_DATE			0x111u

#define CKA_MODULUS				0x120u

#define CKA_MODULUS_BITS		0x121u

#define CKA_PUBLIC_EXPONENT		0x122u

#define CKA_PRIVATE_EXPONENT	0x123u

#define CKA_PRIME_1				0x124u

#define CKA_PRIME_2				0x125u

#define CKA_EXPONENT_1			0x126u

#define CKA_EXPONENT_2			0x127u

#define CKA_COEFFICIENT			0x128u

#define CKA_PRIME				0x130u

#define CKA_SUBPRIME			0x131u

#define CKA_BASE				0x132u

#define CKA_VALUE_BITS			0x160u

#define CKA_VALUE_LEN			0x161u

#define CKA_VENDOR_DEFINED		0x8000u

#define CKA_FILE_ID_PRIVATE_KEY	0x8001u			// reserved for internal use

#define CKA_FILE_ID_PUBLIC_KEY	0x8002u			// reserved for internal use

#define CKA_CERTIFICATE_PENDING	0x8003u

#define CKA_DATAKEY_KEY_TYPE	0x8004u

#define CKA_FILE_ID_PARAMETER	0x8005u			// reserved for internal use

#define CKA_FILE_ID_SECRET_KEY	0x8006u	        // reserved for internal use  



// CKA_DATAKEY_KEY_TYPE values

//

#define CKKD_OTHER				0

#define CKKD_ESCROWED_EXCHANGE	1

#define CKKD_SFS_SIGNING		2



// Attribute

//

typedef struct CK_ATTRIBUTE

{

	CK_ATTRIBUTE_TYPE type;

	CK_VOID_PTR pValue;

	CK_USHORT usValueLen;

} CK_ATTRIBUTE;

typedef CK_ATTRIBUTE * CK_ATTRIBUTE_PTR;



// Date

//

typedef struct CK_DATE

{

	CK_CHAR year[4];

	CK_CHAR month[2];

	CK_CHAR day[2];

} CK_DATE;



// Mechanisms ------------------------------------------------------------------



// Mechanism type

//

typedef CK_USHORT CK_MECHANISM_TYPE;

typedef CK_MECHANISM_TYPE * CK_MECHANISM_TYPE_PTR;



// Mechanism types

//

#define CKM_RSA_PKCS_KEY_PAIR_GEN	0x0u

#define CKM_RSA_PKCS				0x1u

#define CKM_RSA_9796				0x2u

#define CKM_RSA_X_509				0x3u

#define CKM_DSA_KEY_PAIR_GEN		0x10u

#define CKM_DSA						0x11u

#define CKM_DH_PKCS_KEY_PAIR_GEN	0x20u

#define CKM_DH_PKCS_DERIVE			0x21u

#define CKM_RC2_KEY_GEN				0x100u

#define CKM_RC2_ECB					0x101u

#define CKM_RC2_CBC					0x102u

#define CKM_RC2_MAC					0x103u

#define CKM_RC4_KEY_GEN				0x110u

#define CKM_RC4						0x111u

#define CKM_DES_KEY_GEN				0x120u

#define CKM_DES_ECB					0x121u

#define CKM_DES_CBC					0x122u

#define CKM_DES_MAC					0x123u

#define CKM_DES2_KEY_GEN			0x130u

#define CKM_DES3_KEY_GEN			0x131u

#define CKM_DES3_ECB				0x132u

#define CKM_DES3_CBC				0x133u

#define CKM_DES3_MAC				0x134u

#define CKM_MD2						0x200u

#define CKM_MD5						0x210u

#define CKM_SHA_1					0x220u

#define CKM_VENDOR_DEFINED			0x8000u



// Mechanism

//

typedef struct CK_MECHANISM

{

	CK_MECHANISM_TYPE mechanism;

	CK_VOID_PTR pParameter;

	CK_USHORT usParameterLen;

} CK_MECHANISM;

typedef CK_MECHANISM * CK_MECHANISM_PTR;



// Mechanism information

//

typedef struct CK_MECHANISM_INFO

{

	CK_ULONG ulMinKeySize;

	CK_ULONG ulMaxKeySize;

	CK_FLAGS flags;

} CK_MECHANISM_INFO;

typedef CK_MECHANISM_INFO * CK_MECHANISM_INFO_PTR;



// Mechanism information flags

//

#define CKF_HW 			0x1u		// TRUE if mechanism performed by device

#define CKF_EXTENSION 	0x8000u		// TRUE if an extension to the flags



// RC2 CBC parameters

//

typedef struct CK_RC2_CBC_PARAMS

{

	CK_USHORT usEffectiveBits;

	CK_BYTE iv[8];

} CK_RC2_CBC_PARAMS;

typedef CK_RC2_CBC_PARAMS * CK_RC2_CBC_PARAMS_PTR;



#ifdef WIN32

#define CK_ENTRY	__declspec(dllexport)

#else

#define CK_ENTRY    _export _far _pascal

#endif 



// Return values

//

typedef CK_USHORT CK_RV;			// Cryptoki function return value

#define CKR_OK									0x0u

#define CKR_CANCEL								0x1u

#define CKR_HOST_MEMORY							0x2u

#define CKR_SLOT_ID_INVALID						0x3u

#define CKR_FLAGS_INVALID						0x4u

#define CKR_ATTRIBUTE_READ_ONLY					0x10u

#define CKR_ATTRIBUTE_SENSITIVE					0x11u

#define CKR_ATTRIBUTE_TYPE_INVALID				0x12u

#define CKR_ATTRIBUTE_VALUE_INVALID				0x13u

#define CKR_DATA_INVALID						0x20u

#define CKR_DATA_LEN_RANGE						0x21u

#define CKR_DEVICE_ERROR						0x30u

#define CKR_DEVICE_MEMORY						0x31u

#define CKR_DEVICE_REMOVED						0x32u

#define CKR_ENCRYPTED_DATA_INVALID				0x40u

#define CKR_ENCRYPTED_DATA_LEN_RANGE			0x41u

#define CKR_FUNCTION_CANCELED					0x50u

#define CKR_FUNCTION_NOT_PARALLEL				0x51u

#define CKR_FUNCTION_PARALLEL					0x52u

#define CKR_KEY_HANDLE_INVALID					0x60u

#define CKR_KEY_SENSITIVE						0x61u

#define CKR_KEY_SIZE_RANGE						0x62u

#define CKR_KEY_TYPE_INCONSISTENT				0x63u

#define CKR_MECHANISM_INVALID					0x70u

#define CKR_MECHANISM_PARAM_INVALID				0x71u

#define CKR_OBJECT_CLASS_INCONSISTENT			0x80u

#define CKR_OBJECT_CLASS_INVALID				0x81u

#define CKR_OBJECT_HANDLE_INVALID				0x82u

#define CKR_OPERATION_ACTIVE					0x90u

#define CKR_OPERATION_NOT_INITIALIZED			0x91u

#define CKR_PIN_INCORRECT						0xa0u

#define CKR_PIN_INVALID							0xa1u

#define CKR_PIN_LEN_RANGE						0xa2u

#define CKR_SESSION_CLOSED						0xb0u

#define CKR_SESSION_COUNT						0xb1u

#define CKR_SESSION_EXCLUSIVE_EXISTS			0xb2u

#define CKR_SESSION_HANDLE_INVALID				0xb3u

#define CKR_SESSION_PARALLEL_NOT_SUPPORTED		0xb4u

#define CKR_SESSION_READ_ONLY					0xb5u

#define CKR_SESSION_EXISTS						0xb6u

#define CKR_SIGNATURE_INVALID					0xc0u

#define CKR_SIGNATURE_LEN_RANGE					0xc1u

#define CKR_TEMPLATE_INCOMPLETE					0xd0u

#define CKR_TEMPLATE_INCONSISTENT				0xd1u

#define CKR_TOKEN_NOT_PRESENT					0xe0u

#define CKR_TOKEN_NOT_RECOGNIZED				0xe1u

#define CKR_TOKEN_WRITE_PROTECTED				0xe2u

#define CKR_UNWRAPPING_KEY_HANDLE_INVALID		0xf0u

#define CKR_UNWRAPPING_KEY_SIZE_RANGE			0xf1u

#define CKR_UNWRAPPING_KEY_TYPE_INCONSISTENT	0xf2u

#define CKR_USER_ALREADY_LOGGED_IN				0x100u

#define CKR_USER_NOT_LOGGED_IN					0x101u

#define CKR_USER_PIN_NOT_INITIALIZED			0x102u

#define CKR_USER_TYPE_INVALID					0x103u

#define CKR_WRAPPED_KEY_INVALID					0x110u

#define CKR_WRAPPED_KEY_LEN_RANGE				0x112u

#define CKR_WRAPPING_KEY_HANDLE_INVALID			0x113u

#define CKR_WRAPPING_KEY_SIZE_RANGE				0x114u

#define CKR_WRAPPING_KEY_TYPE_INCONSISTENT		0x115u

#define CKR_RANDOM_SEED_NOT_SUPPORTED			0x120u

#define CKR_VENDOR_DEFINED						0x8000u

#define CKR_NOT_IMPLEMENTED						0x8001u

#define CKR_INTERNAL_ERROR						0x8002u

#define CKR_SO_NOT_LOGGED_IN					0x8003u

#define CKR_CRYPTO_ERROR						0x8004u

#define CKR_FILE_NOT_FOUND						0x8005u

#define CKR_DSO_NOT_LOGGED_IN					0x8006u

#define CKR_PIN_LOCKED							0x8007u

#define CKR_USER_PIN_ALREADY_INITIALIZED		0x8008u

#define CKR_PIN_INVALID_TYPE					0x8009u





#ifdef WIN32

typedef	CK_RV CK_ENTRY (*Notify)(CK_SESSION_HANDLE hSession, CK_NOTIFICATION event,

	CK_VOID_PTR pApplication);

#else // WIN16

typedef	CK_RV (CK_ENTRY *Notify)(CK_SESSION_HANDLE hSession, CK_NOTIFICATION event,

	CK_VOID_PTR pApplication);

#endif



#endif	// CRYPTOKI_VERSION



// Types for Datakey extensions ------------------------------------------------

//

// Answer To Reset file

// NOTE that none of the character arrays are null-terminated

//

typedef struct AnswerToReset

{

	CK_BYTE length;					// length of data to send on token powerup

	CK_BYTE atr[9];					// TS - TD

	CK_BYTE historical[16];			// historical bytes - as needed

	CK_BYTE reserved[2];			// reserved for future use

} AnswerToReset;



// Bit lengths of keys used by the token.

// NOTE that the values are 256 bit multipliers.

// Actual key lengths are multiplier * 256.

//

enum {keyMultiplier = 256};

typedef struct KeyLength

{

	CK_BYTE dsaSignature;			// DSA Signature

	CK_BYTE rsaSignature;			// RSA Signature

	CK_BYTE rsaExchange;			// RSA Exchange

	CK_BYTE reserved;				// reserved for future use

} KeyLength;



// Datakey token configuration information

//

typedef struct DatakeyInfo

{

	CK_CHAR serialNumber[8];		// serial number

	CK_CHAR	osVersion[16];			// chip OS version

	CK_CHAR vendor[14];				// vendor

	CK_CHAR manufacturingDate[8];	// manufacturing date

	CK_BYTE dkisMajor;				// major version number

	CK_BYTE dkisMinor;				// minor version number

	CK_BYTE maxPinLength;			// maximum PIN length

	CK_BYTE export;					// TRUE if export version

	CK_BYTE reserved[16];			// reserved for future use



} DatakeyInfo;



// Datakey security policy information

//

typedef struct SecurityPolicyInfo

{

	CK_BYTE ckMajor;				// cryptoki major version number

	CK_BYTE ckMinor;				// cryptoki minor version number

	CK_BYTE spifMajor;				// SPIF major version number

	CK_BYTE spifMinor;				// SPIF minor version number

	CK_BYTE soapMajor;				// SO app major version number

	CK_BYTE soapMinor;				// SO app minor version number

	CK_BYTE maxLoginAttempts;		// maximum login attempts

	CK_BYTE minPinLength;			// minimum pin length

	CK_BYTE pinTimer;				// number of minutes PIN is valid

	KeyLength algid;				// key length multiplier (x256)

	CK_BYTE reserved[37];			// reserved for future use

} SecurityPolicyInfo;



// Note that the 330 chip has a different configuration file format. When the 330

// code is added, Hazem may want to add a structure to work with both chips.

//

//	Dkccos Configuration file format.

//

typedef struct ConfigFile

{

	CK_BYTE rsaPrivEnableSize;

	CK_BYTE rsaSignEnableSize;

	CK_BYTE dsaEnableSize;

	CK_BYTE dhEnableSize[2];

	CK_BYTE pubKeyFormat;

	CK_BYTE symKeyEnable;

	CK_BYTE soAuth;

	CK_BYTE userAuth;

	CK_BYTE exfEnable[2];

	CK_BYTE idleAllow ;

	CK_BYTE RFU[4];

	CK_BYTE configurationMask[16];

} ConfigFile;

// Structure of data in Dkccos GetStatus response. 

//

typedef struct DkccosStatus

{

	CK_BYTE cardState;

	CK_BYTE remFileSpace[2];

	CK_BYTE firmwareVersion[2];

	CK_BYTE exfStatus;

	CK_BYTE exfNumber[2];

	CK_BYTE errorStatus;

	CK_BYTE RFU[4];

	ConfigFile configData;

} DkccosStatus;





//

// PIN Types

//

#define	CK_PIN_ASCII	0

#define CK_PIN_BINARY	1

#define CK_PIN_BCD		2

/*

typedef enum CK_PIN_TYPE

{

	CK_PIN_ASCII,		// PIN format is ascii, i.e. 1234 = 0x31 0x32 0x33 0x34

	CK_PIN_BINARY,		// PIN format is bin, i.e. 1234 = 0x01 0x02 0x03 0x04

	CK_PIN_BCD			// PIN format is BCD, i.e. 1234 = 0x12 0x34

} CK_PIN_TYPE;

*/

// User Information

//

typedef struct UserInfo

{

	CK_CHAR label[32];			// same as label parameter to C_InitToken

	CK_BYTE pinTimer;			// upper limit is DatakeyInfo.pinTimer

	CK_BYTE pinType;			// Format of pin stored on token

	CK_BYTE reserved[32];		// reserved for future use

} UserInfo;



#ifdef __cplusplus

extern "C"

{

#endif // __cplusplus



// The Cryptoki interface -----------------------------------------------------

//



#if CRYPTOKI_VERSION < 2



	// General purpose -----------------

	//

	CK_RV CK_ENTRY C_Initialize(CK_VOID_PTR pReserved);

	CK_RV CK_ENTRY C_Finalize(CK_VOID_PTR pReserved);

	CK_RV CK_ENTRY C_GetInfo(CK_INFO_PTR pInfo);



	// Slot and token management -------

	//

	CK_RV CK_ENTRY C_GetSlotList(

		CK_BBOOL tokenPresent,

		CK_SLOT_ID_PTR pSlotList,

		CK_USHORT_PTR pusCount);



	CK_RV CK_ENTRY C_GetSlotInfo(

		CK_SLOT_ID slotID,

		CK_SLOT_INFO_PTR pInfo);



	CK_RV CK_ENTRY C_GetTokenInfo(

		CK_SLOT_ID slotID,

		CK_TOKEN_INFO_PTR pInfo);



	CK_RV CK_ENTRY C_GetMechanismList(

		CK_SLOT_ID slotID,

		CK_MECHANISM_TYPE_PTR pMechanismList,

		CK_USHORT_PTR pusCount);



	CK_RV CK_ENTRY C_GetMechanismInfo(

		CK_SLOT_ID slotID,

		CK_MECHANISM_TYPE type,

		CK_MECHANISM_INFO_PTR pInfo);



	CK_RV CK_ENTRY C_InitToken(

		CK_SLOT_ID slotID,

		CK_CHAR_PTR pPin,

		CK_USHORT usPinLen,

		CK_CHAR_PTR pLabel);



	CK_RV CK_ENTRY C_InitPIN(

		CK_SESSION_HANDLE hSession,

		CK_CHAR_PTR pPin,

		CK_USHORT usPinLen);



	CK_RV CK_ENTRY C_SetPIN(

		CK_SESSION_HANDLE hSession,

		CK_CHAR_PTR pOldPin,

		CK_USHORT usOldLen,

		CK_CHAR_PTR pNewPin,

		CK_USHORT usNewLen);



	// Session Management --------------

	//

	CK_RV CK_ENTRY C_OpenSession(

		CK_SLOT_ID slotID,

		CK_FLAGS flags,

		CK_VOID_PTR pApplication,

		Notify Notify,

		CK_SESSION_HANDLE_PTR phSession);



	CK_RV CK_ENTRY C_CloseSession(CK_SESSION_HANDLE hSession);



	CK_RV CK_ENTRY C_CloseAllSessions(CK_SLOT_ID slotID);



	CK_RV CK_ENTRY C_GetSessionInfo(

		CK_SESSION_HANDLE hSession,

		CK_SESSION_INFO_PTR pInfo);



	CK_RV CK_ENTRY C_Login(

		CK_SESSION_HANDLE hSession,

		CK_USER_TYPE userType,

		CK_CHAR_PTR pPin,

		CK_USHORT usPinLen);



	CK_RV CK_ENTRY C_Logout(CK_SESSION_HANDLE hSession);



	// Object Management ---------------

	//

	CK_RV CK_ENTRY C_CreateObject(

		CK_SESSION_HANDLE hSession,

		CK_ATTRIBUTE_PTR pTemplate,

		CK_USHORT usCount,

		CK_OBJECT_HANDLE_PTR phObject);



	CK_RV CK_ENTRY C_CopyObject(

		CK_SESSION_HANDLE hSession,

		CK_OBJECT_HANDLE hObject,

		CK_ATTRIBUTE_PTR pTemplate,

		CK_USHORT usCount,

		CK_OBJECT_HANDLE_PTR phNewObject);



	CK_RV CK_ENTRY C_DestroyObject(

		CK_SESSION_HANDLE hSession,

		CK_OBJECT_HANDLE hObject);



	CK_RV CK_ENTRY C_GetObjectSize(

		CK_SESSION_HANDLE hSession,

		CK_OBJECT_HANDLE hObject,

		CK_USHORT_PTR pusSize);



	CK_RV CK_ENTRY C_GetAttributeValue(

		CK_SESSION_HANDLE hSession,

		CK_OBJECT_HANDLE hObject,

		CK_ATTRIBUTE_PTR pTemplate,

		CK_USHORT usCount);



	CK_RV CK_ENTRY C_SetAttributeValue(

		CK_SESSION_HANDLE hSession,

		CK_OBJECT_HANDLE hObject,

		CK_ATTRIBUTE_PTR pTemplate,

		CK_USHORT usCount);



	CK_RV CK_ENTRY C_FindObjectsInit(

		CK_SESSION_HANDLE hSession,

		CK_ATTRIBUTE_PTR pTemplate,

		CK_USHORT usCount);



	CK_RV CK_ENTRY C_FindObjects(

		CK_SESSION_HANDLE hSession,

		CK_OBJECT_HANDLE_PTR phObject,

		CK_USHORT usMaxObjectCount,

		CK_USHORT_PTR pusObjectCount);



	// Encryption and decryption -------

	//

	CK_RV CK_ENTRY C_EncryptInit(

		CK_SESSION_HANDLE hSession,

		CK_MECHANISM_PTR pMechanism,

		CK_OBJECT_HANDLE hKey);



	CK_RV CK_ENTRY C_Encrypt(

		CK_SESSION_HANDLE hSession,

		CK_BYTE_PTR pData,

		CK_USHORT usDataLen,

		CK_BYTE_PTR pEncryptedData,

		CK_USHORT_PTR pusEncryptedDataLen);



	CK_RV CK_ENTRY C_EncryptUpdate(

		CK_SESSION_HANDLE hSession,

		CK_BYTE_PTR pPart,

		CK_USHORT usPartLen,

		CK_BYTE_PTR pEncryptedPart,

		CK_USHORT_PTR pusEncryptedPartLen);



	CK_RV CK_ENTRY C_EncryptFinal(

		CK_SESSION_HANDLE hSession,

		CK_BYTE_PTR pLastEncryptedPart,

		CK_USHORT_PTR pusLastEncryptedPartLen);



	CK_RV CK_ENTRY C_DecryptInit(

		CK_SESSION_HANDLE hSession,

		CK_MECHANISM_PTR pMechanism,

		CK_OBJECT_HANDLE hKey);



	CK_RV CK_ENTRY C_Decrypt(

		CK_SESSION_HANDLE hSession,

		CK_BYTE_PTR pEncryptedData,

		CK_USHORT usEncryptedDataLen,

		CK_BYTE_PTR pData,

		CK_USHORT_PTR pusDataLen);



	CK_RV CK_ENTRY C_DecryptUpdate(

		CK_SESSION_HANDLE hSession,

		CK_BYTE_PTR pEncryptedPart,

		CK_USHORT usEncryptedPartLen,

		CK_BYTE_PTR pPart,

		CK_USHORT_PTR pusPartLen);



	CK_RV CK_ENTRY C_DecryptFinal(

		CK_SESSION_HANDLE hSession,

		CK_BYTE_PTR pLastPart,

		CK_USHORT_PTR pusLastPartLen);



	// Message digesting ---------------

	//

	CK_RV CK_ENTRY C_DigestInit(

		CK_SESSION_HANDLE hSession,

		CK_MECHANISM_PTR pMechanism);



	CK_RV CK_ENTRY C_Digest(

		CK_SESSION_HANDLE hSession,

		CK_BYTE_PTR pData,

		CK_USHORT usDataLen,

		CK_BYTE_PTR pDigest,

		CK_USHORT_PTR pusDigestLen);



	CK_RV CK_ENTRY C_DigestUpdate(

		CK_SESSION_HANDLE hSession,

		CK_BYTE_PTR pPart,

		CK_USHORT usPartLen);



	CK_RV CK_ENTRY C_DigestFinal(

		CK_SESSION_HANDLE hSession,

		CK_BYTE_PTR pDigest,

		CK_USHORT_PTR pusDigestLen);



	// Signature and verification ------

	//

	CK_RV CK_ENTRY C_SignInit(

		CK_SESSION_HANDLE hSession,

		CK_MECHANISM_PTR pMechanism,

		CK_OBJECT_HANDLE hKey);



	CK_RV CK_ENTRY C_Sign(

		CK_SESSION_HANDLE hSession,

		CK_BYTE_PTR pData,

		CK_USHORT usDataLen,

		CK_BYTE_PTR pSignature,

		CK_USHORT_PTR pusSignatureLen);



	CK_RV CK_ENTRY C_SignUpdate(

		CK_SESSION_HANDLE hSession,

		CK_BYTE_PTR pPart,

		CK_USHORT usPartLen);



	CK_RV CK_ENTRY C_SignFinal(

		CK_SESSION_HANDLE hSession,

		CK_BYTE_PTR pSignature,

		CK_USHORT_PTR pusSignatureLen);



	CK_RV CK_ENTRY C_SignRecoverInit(

		CK_SESSION_HANDLE hSession,

		CK_MECHANISM_PTR pMechanism,

		CK_OBJECT_HANDLE hKey);



	CK_RV CK_ENTRY C_SignRecover(

		CK_SESSION_HANDLE hSession,

		CK_BYTE_PTR pData,

		CK_USHORT usDataLen,

		CK_BYTE_PTR pSignature,

		CK_USHORT_PTR pusSignatureLen);



	CK_RV CK_ENTRY C_VerifyInit(

		CK_SESSION_HANDLE hSession,

		CK_MECHANISM_PTR pMechanism,

		CK_OBJECT_HANDLE hKey);



	CK_RV CK_ENTRY C_Verify(

		CK_SESSION_HANDLE hSession,

		CK_BYTE_PTR pData,

		CK_USHORT usDataLen,

		CK_BYTE_PTR pSignature,

		CK_USHORT usSignatureLen);



	CK_RV CK_ENTRY C_VerifyUpdate(

		CK_SESSION_HANDLE hSession,

		CK_BYTE_PTR pPart,

		CK_USHORT usPartLen);



	CK_RV CK_ENTRY C_VerifyFinal(

		CK_SESSION_HANDLE hSession,

		CK_BYTE_PTR pSignature,

		CK_USHORT signatureLen);



	CK_RV CK_ENTRY C_VerifyRecoverInit(

		CK_SESSION_HANDLE hSession,

		CK_MECHANISM_PTR pMechanism,

		CK_OBJECT_HANDLE hKey);



	CK_RV CK_ENTRY C_VerifyRecover(

		CK_SESSION_HANDLE hSession,

		CK_BYTE_PTR pSignature,

		CK_USHORT signatureLen,

		CK_BYTE_PTR pData,

		CK_USHORT_PTR pusDataLen);



	// Key Management ------------------

	//

	CK_RV CK_ENTRY C_GenerateKey(

		CK_SESSION_HANDLE hSession,

		CK_MECHANISM_PTR pMechanism,

		CK_ATTRIBUTE_PTR pTemplate,

		CK_USHORT usCount,

		CK_OBJECT_HANDLE_PTR phKey);



	CK_RV CK_ENTRY C_GenerateKeyPair(

		CK_SESSION_HANDLE hSession,

		CK_MECHANISM_PTR pMechanism,

		CK_ATTRIBUTE_PTR pPublicKeyTemplate,

		CK_USHORT usPublicKeyAttributeCount,

		CK_ATTRIBUTE_PTR pPrivateKeyTemplate,

		CK_USHORT usPrivateKeyAttributeCount,

		CK_OBJECT_HANDLE_PTR phPrivateKey,

		CK_OBJECT_HANDLE_PTR phPublicKey);



	CK_RV CK_ENTRY C_WrapKey(

		CK_SESSION_HANDLE hSession,

		CK_MECHANISM_PTR pMechanism,

		CK_OBJECT_HANDLE hWrappingKey,

		CK_OBJECT_HANDLE hKey,

		CK_BYTE_PTR pWrappedKey,

		CK_USHORT_PTR pusWrappedKeyLen);



	CK_RV CK_ENTRY C_UnwrapKey(

		CK_SESSION_HANDLE hSession,

		CK_MECHANISM_PTR pMechanism,

		CK_OBJECT_HANDLE hUnwrappingKey,

		CK_BYTE_PTR pWrappedKey,

		CK_USHORT usWrappedKeyLen,

		CK_ATTRIBUTE_PTR pTemplate,

		CK_USHORT usAttributeCount,

		CK_OBJECT_HANDLE_PTR phKey);



	CK_RV CK_ENTRY C_DeriveKey(

		CK_SESSION_HANDLE hSession,

		CK_MECHANISM_PTR pMechanism,

		CK_OBJECT_HANDLE hBaseKey,

		CK_ATTRIBUTE_PTR pTemplate,

		CK_USHORT usAttributeCount,

		CK_OBJECT_HANDLE_PTR phKey);



	// Random number generation --------

	//

	CK_RV CK_ENTRY C_SeedRandom(

		CK_SESSION_HANDLE hSession,

		CK_BYTE_PTR pSeed,

		CK_USHORT usSeedLen);



	CK_RV CK_ENTRY C_GenerateRandom(

		CK_SESSION_HANDLE hSession,

		CK_BYTE_PTR pRandomData,

		CK_USHORT usRandomLen);



	// Parallel function management ----

	//

	CK_RV CK_ENTRY C_GetFunctionStatus(CK_SESSION_HANDLE hSession);

	CK_RV CK_ENTRY C_CancelFunction(CK_SESSION_HANDLE hSession);



#endif	// CRYPTOKI_VERSION



	// Datakey extensions --------------

	//

	CK_RV CK_ENTRY D_GetSlotSN(CK_SLOT_ID slotID, BYTE *richardDale);



	CK_RV CK_ENTRY D_GetATR(CK_SESSION_HANDLE hSession, AnswerToReset *atr);



	CK_RV CK_ENTRY D_GetUIS(CK_SESSION_HANDLE hSession, UserInfo *userInfo);

	CK_RV CK_ENTRY D_SetUIS(CK_SESSION_HANDLE hSession, UserInfo *userInfo);



	CK_RV CK_ENTRY D_GetDKIS(CK_SESSION_HANDLE hSession, DatakeyInfo *datakeyInfo);



	CK_RV CK_ENTRY D_GetSPIF(CK_SESSION_HANDLE hSession,

		SecurityPolicyInfo *securityPolicyInfo);

	CK_RV CK_ENTRY D_SetSPIF(CK_SESSION_HANDLE hSession,       

		SecurityPolicyInfo *securityPolicyInfo);



	CK_RV CK_ENTRY _stdcall D_GetDkStatus(CK_SLOT_ID slotId, DkccosStatus *spinfo);



#if CRYPTOKI_VERSION > 1

	CK_RV CK_ENTRY D_GetSeed(CK_SESSION_HANDLE hSession,

		CK_BYTE_PTR seed, CK_ULONG seedSize);

	CK_RV CK_ENTRY D_GetAllAttributeValues(CK_SESSION_HANDLE hSession,CK_OBJECT_HANDLE hObject,CK_ATTRIBUTE_PTR pTemplate,CK_ULONG ulCount);

	CK_RV CK_ENTRY D_GetAttributeCount(CK_SESSION_HANDLE hSession,CK_OBJECT_HANDLE hObject,CK_ULONG& ulCount);

	CK_RV CK_ENTRY _stdcall D_ChangeConfigFile(CK_SLOT_ID slotId, ConfigFile *configfile, CK_BYTE *passphrase,CK_ULONG uspassphrase);

#else

	CK_RV CK_ENTRY D_GetSeed(CK_SESSION_HANDLE hSession,

		CK_BYTE_PTR seed, CK_USHORT seedSize);

	CK_RV CK_ENTRY D_GetAllAttributeValues(CK_SESSION_HANDLE hSession,CK_OBJECT_HANDLE hObject,CK_ATTRIBUTE_PTR pTemplate,CK_USHORT usCount);

//	CK_RV CK_ENTRY D_GetAttributeCount(CK_SESSION_HANDLE hSession,CK_OBJECT_HANDLE hObject,CK_USHORT& usCount);

	CK_RV CK_ENTRY _stdcall D_ChangeConfigFile(CK_SLOT_ID slotId, ConfigFile *configfile, CK_BYTE *passphrase,CK_USHORT uspassphrase);

#endif	// CRYPTOKI_VERSION



	void CK_ENTRY D_SetTraceMask(CK_ULONG mask);



#ifdef __cplusplus

}

#endif // __cplusplus



#if CRYPTOKI_VERSION < 2

#ifdef WIN32

#pragma pack(pop,cryptoki)

#endif 

#endif



#endif	// CRYPTOKI_H

